<?php
$auth_token = $_COOKIE['rcp_auth'] ?? null;
if (!$auth_token) {
    header("Location: login.php");
    exit;
}

try {
    require_once __DIR__ . '/../includes/config.php';
    require_once __DIR__ . '/../includes/database.php';
    
    $pdo = Database::getInstance()->getConnection();
    
    $stmt = $pdo->prepare("SELECT session_data FROM user_sessions WHERE session_id = ? AND expires_at > NOW()");
    $stmt->execute([$auth_token]);
    $session_data = $stmt->fetchColumn();
    
    if (!$session_data) {
        header("Location: login.php");
        exit;
    }
    
    $user = json_decode($session_data, true);
    
} catch (Throwable $t) {
    die("Error: " . $t->getMessage());
}

$hasNavbar = is_file(__DIR__ . '/../includes/navbar.php');
$hasFooter = is_file(__DIR__ . '/../includes/footer.php');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Smart Serve Pro - System Settings</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        body { background: #f8f9fa; padding-top: 0; }
        .card { border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); border: none; margin-bottom: 20px; }
        .card-header { background: #8B0000 !important; color: white; border-bottom: none; padding: 15px 20px; }
        .btn-primary { background-color: #8B0000; border-color: #8B0000; color: white; }
        .btn-primary:hover { background-color: #600000; border-color: #600000; color: white; }
        .btn-secondary { background-color: #6c757d; border-color: #6c757d; }
        .btn-outline-primary { border-color: #8B0000; color: #8B0000; }
        .btn-outline-primary:hover { background-color: #8B0000; color: white; }
        .btn-outline-danger { border-color: #dc3545; color: #dc3545; }
        .btn-outline-danger:hover { background-color: #dc3545; border-color: #dc3545; color: white; }
        #logoPreview { border: 2px dashed #dee2e6; padding: 10px; border-radius: 8px; max-height: 100px; max-width: 200px; }
    </style>
</head>
<body>
    <?php if ($hasNavbar) include __DIR__ . '/../includes/navbar.php'; ?>

    <main class="container mt-4">
        <div class="row mb-4">
            <div class="col-12">
                <h1 class="h3 mb-1">System Settings</h1>
                <p class="text-muted">Manage application settings and company information</p>
            </div>
        </div>

        <div class="row">
            <div class="col-lg-6">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="fas fa-building me-2"></i>Company Information</h5>
                    </div>
                    <div class="card-body">
                        <form id="companyForm">
                            <div class="mb-3">
                                <label for="companyName" class="form-label">Company Name *</label>
                                <input type="text" class="form-control" id="companyName" name="company_name" required>
                                <div class="form-text">This will be displayed in the footer as "Registered to"</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="companyAddress" class="form-label">Address</label>
                                <textarea class="form-control" id="companyAddress" name="company_address" rows="3"></textarea>
                            </div>
                            
                            <div class="row g-2">
                                <div class="col-md-6">
                                    <label for="companyPhone" class="form-label">Phone</label>
                                    <input type="text" class="form-control" id="companyPhone" name="company_phone">
                                </div>
                                <div class="col-md-6">
                                    <label for="companyEmail" class="form-label">Email</label>
                                    <input type="email" class="form-control" id="companyEmail" name="company_email">
                                </div>
                            </div>
                            
                            <div class="row g-2 mt-3">
                                <div class="col-md-6">
                                    <label for="baseCurrency" class="form-label">Base Currency</label>
                                    <select class="form-select" id="baseCurrency" name="base_currency">
                                        <option value="ZAR">ZAR - South African Rand</option>
                                        <option value="USD">USD - US Dollar</option>
                                        <option value="EUR">EUR - Euro</option>
                                        <option value="GBP">GBP - British Pound</option>
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <label for="decimalPlaces" class="form-label">Decimal Places</label>
                                    <select class="form-select" id="decimalPlaces" name="decimal_places">
                                        <option value="2">2 Decimal Places</option>
                                        <option value="0">Whole Numbers</option>
                                        <option value="3">3 Decimal Places</option>
                                    </select>
                                </div>
                            </div>
                            
                            <button type="submit" class="btn btn-primary mt-3">
                                <i class="fas fa-save me-2"></i>Save Company Details
                            </button>
                        </form>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-6">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="fas fa-image me-2"></i>Branding</h5>
                    </div>
                    <div class="card-body">
                        <div class="text-center mb-4">
                            <img id="logoPreview" src="<?= BASE_PATH ?>assets/images/logo-placeholder.png" alt="Company Logo" class="img-fluid">
                            <div class="mt-2 text-muted small">Logo Preview</div>
                        </div>
                        
                        <form id="logoForm">
                            <div class="mb-3">
                                <label for="logoUpload" class="form-label">Upload Logo</label>
                                <input type="file" class="form-control" id="logoUpload" name="logo" accept=".png,.jpg,.jpeg,.gif,.webp">
                                <div class="form-text">Supported formats: PNG, JPG, GIF, WebP. Max size: 2MB.</div>
                            </div>
                            
                            <button type="submit" class="btn btn-secondary w-100">
                                <i class="fas fa-upload me-2"></i>Upload Logo
                            </button>
                        </form>
                        
                        <div class="mt-3">
                            <button type="button" class="btn btn-outline-danger w-100" id="removeLogoBtn">
                                <i class="fas fa-trash me-2"></i>Remove Logo
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="card mt-4">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="fas fa-cog me-2"></i>System Preferences</h5>
                    </div>
                    <div class="card-body">
                        <form id="preferencesForm">
                            <div class="mb-3">
                                <label for="timezone" class="form-label">Timezone</label>
                                <select class="form-select" id="timezone" name="timezone">
                                    <option value="Africa/Johannesburg">Africa/Johannesburg</option>
                                    <option value="UTC">UTC</option>
                                </select>
                            </div>
                            
                            <div class="mb-3">
                                <label for="dateFormat" class="form-label">Date Format</label>
                                <select class="form-select" id="dateFormat" name="date_format">
                                    <option value="Y-m-d">YYYY-MM-DD</option>
                                    <option value="d/m/Y">DD/MM/YYYY</option>
                                </select>
                            </div>
                            
                            <button type="submit" class="btn btn-outline-primary">
                                <i class="fas fa-save me-2"></i>Save Preferences
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <script>
    class SettingsManager {
        constructor() {
            this.basePath = '<?= BASE_PATH ?>';
            this.init();
        }

        init() {
            this.setupEventListeners();
            this.loadSettings();
        }

        setupEventListeners() {
            document.getElementById('companyForm').addEventListener('submit', (e) => {
                e.preventDefault(); this.saveCompanySettings();
            });
            document.getElementById('logoForm').addEventListener('submit', (e) => {
                e.preventDefault(); this.uploadLogo();
            });
            document.getElementById('preferencesForm').addEventListener('submit', (e) => {
                e.preventDefault(); this.savePreferences();
            });
            document.getElementById('removeLogoBtn').addEventListener('click', () => {
                this.removeLogo();
            });
            document.getElementById('logoUpload').addEventListener('change', (e) => {
                this.previewLogo(e.target.files[0]);
            });
        }

        async loadSettings() {
            try {
                const companyResponse = await fetch('../api/settings.php?action=get_company');
                const companyData = await companyResponse.json();
                if (companyData.success) this.populateCompanyForm(companyData.settings);

                const prefsResponse = await fetch('../api/settings.php?action=get_preferences');
                const prefsData = await prefsResponse.json();
                if (prefsData.success) this.populatePreferencesForm(prefsData.settings);

                this.showAlert('Settings loaded!', 'success');
            } catch (error) {
                this.showAlert('Error loading settings', 'danger');
            }
        }

        populateCompanyForm(settings) {
            document.getElementById('companyName').value = settings.company_name || '';
            document.getElementById('companyAddress').value = settings.company_address || '';
            document.getElementById('companyPhone').value = settings.company_phone || '';
            document.getElementById('companyEmail').value = settings.company_email || '';
            document.getElementById('baseCurrency').value = settings.base_currency || 'ZAR';
            document.getElementById('decimalPlaces').value = settings.decimal_places || '2';
            
            const logoPreview = document.getElementById('logoPreview');
            if (settings.logo_path && settings.logo_path !== '') {
                logoPreview.src = settings.logo_path + '?t=' + Date.now();
            }
        }

        populatePreferencesForm(settings) {
            document.getElementById('timezone').value = settings.timezone || 'Africa/Johannesburg';
            document.getElementById('dateFormat').value = settings.date_format || 'Y-m-d';
        }

        async saveCompanySettings() {
            const submitBtn = document.querySelector('#companyForm button[type="submit"]');
            const originalText = submitBtn.innerHTML;

            try {
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Saving...';
                submitBtn.disabled = true;

                const formData = new FormData(document.getElementById('companyForm'));
                const response = await fetch('../api/settings.php?action=save_company', {
                    method: 'POST', body: formData
                });

                const data = await response.json();
                if (data.success) {
                    this.showAlert('Company settings saved!', 'success');
                } else {
                    this.showAlert(data.message || 'Error saving settings', 'danger');
                }
            } catch (error) {
                this.showAlert('Error saving settings', 'danger');
            } finally {
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            }
        }

        async savePreferences() {
            const submitBtn = document.querySelector('#preferencesForm button[type="submit"]');
            const originalText = submitBtn.innerHTML;

            try {
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Saving...';
                submitBtn.disabled = true;

                const formData = new FormData(document.getElementById('preferencesForm'));
                const response = await fetch('../api/settings.php?action=save_preferences', {
                    method: 'POST', body: formData
                });

                const data = await response.json();
                if (data.success) {
                    this.showAlert('Preferences saved!', 'success');
                } else {
                    this.showAlert(data.message || 'Error saving preferences', 'danger');
                }
            } catch (error) {
                this.showAlert('Error saving preferences', 'danger');
            } finally {
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            }
        }

        async uploadLogo() {
            const fileInput = document.getElementById('logoUpload');
            const submitBtn = document.querySelector('#logoForm button[type="submit"]');
            const originalText = submitBtn.innerHTML;

            if (!fileInput.files[0]) {
                this.showAlert('Please select a file', 'danger');
                return;
            }

            const file = fileInput.files[0];
            if (file.size > 2 * 1024 * 1024) {
                this.showAlert('File size must be less than 2MB', 'danger');
                return;
            }

            try {
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Uploading...';
                submitBtn.disabled = true;

                const formData = new FormData(document.getElementById('logoForm'));
                const response = await fetch('../api/settings.php?action=upload_logo', {
                    method: 'POST', body: formData
                });

                const data = await response.json();
                if (data.success) {
                    this.showAlert('Logo uploaded!', 'success');
                    document.getElementById('logoPreview').src = data.logo_path + '?t=' + Date.now();
                    document.getElementById('logoForm').reset();
                } else {
                    this.showAlert(data.message || 'Error uploading logo', 'danger');
                }
            } catch (error) {
                this.showAlert('Error uploading logo', 'danger');
            } finally {
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            }
        }

        async removeLogo() {
            if (!confirm('Are you sure you want to remove the company logo?')) return;

            try {
                const response = await fetch('../api/settings.php?action=remove_logo', { method: 'POST' });
                const data = await response.json();
                if (data.success) {
                    this.showAlert('Logo removed!', 'success');
                    document.getElementById('logoPreview').src = this.basePath + 'assets/images/logo-placeholder.png';
                } else {
                    this.showAlert(data.message || 'Error removing logo', 'danger');
                }
            } catch (error) {
                this.showAlert('Error removing logo', 'danger');
            }
        }

        previewLogo(file) {
            if (file) {
                const reader = new FileReader();
                reader.onload = (e) => {
                    document.getElementById('logoPreview').src = e.target.result;
                };
                reader.readAsDataURL(file);
            }
        }

        showAlert(message, type) {
            const existingAlerts = document.querySelectorAll('.alert');
            existingAlerts.forEach(alert => alert.remove());

            const alert = document.createElement('div');
            alert.className = `alert alert-${type} alert-dismissible fade show mt-3`;
            alert.innerHTML = `${message}<button type="button" class="btn-close" data-bs-dismiss="alert"></button>`;

            const container = document.querySelector('.container');
            if (container) container.insertBefore(alert, container.firstChild);

            setTimeout(() => { if (alert.parentNode) alert.remove(); }, 5000);
        }
    }

    document.addEventListener('DOMContentLoaded', function() {
        window.settingsManager = new SettingsManager();
    });
    </script>

    <?php if ($hasFooter) include __DIR__ . '/../includes/footer.php'; ?>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>