<?php
declare(strict_types=1);
/**
 * Login Processing API - DATABASE SESSIONS FIX
 */
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/database.php';

header('Content-Type: application/json');

function dev_error(string $msg): array {
    if (defined('APP_ENV') && APP_ENV === 'development') {
        return ['success' => false, 'message' => $msg];
    }
    return ['success' => false, 'message' => 'Server error. Please try again.'];
}

if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

$email    = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
$password = (string)($_POST['password'] ?? '');

if (!$email || $password === '') {
    echo json_encode(['success' => false, 'message' => 'Email and password are required']);
    exit;
}

try {
    $pdo = Database::getInstance()->getConnection();

    $stmt = $pdo->prepare("
        SELECT u.*, un.name AS unit_name
        FROM users u
        LEFT JOIN units un ON u.unit_id = un.id
        WHERE u.email = ? AND u.status = 'active'
        LIMIT 1
    ");
    $stmt->execute([$email]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user || !isset($user['password']) || !password_verify($password, $user['password'])) {
        try {
            $log = $pdo->prepare("
                INSERT INTO activity_log (unit_id, created_by, type, label, description, ip_address)
                VALUES (?, ?, 'security', 'Failed login attempt', ?, ?)
            ");
            $log->execute([
                $user['unit_id'] ?? null,
                $user['id'] ?? null,
                "Failed login for email: $email",
                $_SERVER['REMOTE_ADDR'] ?? 'Unknown'
            ]);
        } catch (Throwable $ignored) {}
        echo json_encode(['success' => false, 'message' => 'Invalid email or password']);
        exit;
    }

    // DATABASE SESSION AUTHENTICATION
    $auth_token = bin2hex(random_bytes(32));
    
    $user_session_data = [
        'user_id' => $user['id'],
        'user_email' => $user['email'],
        'user_full_name' => $user['full_name'] ?? '',
        'user_role' => $user['role'] ?? 'user',
        'user_status' => $user['status'] ?? 'active',
        'unit_id' => $user['unit_id'] ?? null,
        'unit_name' => $user['unit_name'] ?? null,
        'login_time' => time(),
        'last_activity' => time()
    ];

    // Store session in database
    $sessionStmt = $pdo->prepare("
        INSERT INTO user_sessions (session_id, session_data, expires_at) 
        VALUES (?, ?, DATE_ADD(NOW(), INTERVAL 2 HOUR))
    ");
    $sessionStmt->execute([$auth_token, json_encode($user_session_data)]);

    // Set secure cookie
    setcookie('rcp_auth', $auth_token, [
        'expires' => time() + 7200,
        'path' => '/',
        'domain' => '.kwaggagroep.co.za',
        'secure' => true,
        'httponly' => true,
        'samesite' => 'Lax'
    ]);

    // Log successful login
    try {
        $logStmt = $pdo->prepare("
            INSERT INTO activity_log (unit_id, created_by, type, label, ip_address, user_agent)
            VALUES (?, ?, 'login', 'User logged in', ?, ?)
        ");
        $logStmt->execute([
            $user['unit_id'] ?? null,
            $user['id'] ?? null,
            $_SERVER['REMOTE_ADDR'] ?? 'Unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
        ]);
    } catch (Throwable $ignored) {}

    $redirect = (defined('WEB_BASE') ? WEB_BASE : '/rcp/') . 'pages/dashboard.php';

    echo json_encode([
        'success' => true,
        'message' => 'Login successful',
        'redirect_url' => $redirect,
        'debug' => [
            'auth_token_set' => true,
            'user_id' => $user['id']
        ]
    ]);
    exit;

} catch (Throwable $t) {
    error_log("Login error: " . $t->getMessage());
    echo json_encode(dev_error("Login exception: " . $t->getMessage()));
    exit;
}