<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Include config first
require_once __DIR__ . '/../includes/config.php';

// Use the same session system as navbar.php
$user = null;
$auth_token = $_COOKIE['rcp_auth'] ?? null;

if ($auth_token) {
    try {
        require_once __DIR__ . '/../includes/database.php';
        $pdo = Database::getInstance()->getConnection();
        
        $stmt = $pdo->prepare("SELECT session_data FROM user_sessions WHERE session_id = ? AND expires_at > NOW()");
        $stmt->execute([$auth_token]);
        $session_data = $stmt->fetchColumn();
        
        if ($session_data) {
            $user = json_decode($session_data, true);
        }
    } catch (Throwable $t) {
        error_log("Session error: " . $t->getMessage());
    }
}

// Check if logged in
if (!$user || !isset($user['user_id'])) {
    header('Location: ' . WEB_BASE . 'pages/login.php');
    exit();
}

// Check if superuser
if ($user['user_role'] !== 'superuser') {
    header('Location: ' . WEB_BASE . 'pages/dashboard.php');
    exit();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>User Management - Recipe Control Pro</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="<?= WEB_BASE ?>assets/css/style.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
  <style>
    /* Ensure all text is black */
    body, .card, .card-title, .form-label, .table, .modal-content {
        color: #000000 !important;
    }
    .text-muted {
        color: #6c757d !important;
    }
    .upload-area {
        border: 2px dashed #dee2e6;
        border-radius: 8px;
        padding: 2rem;
        text-align: center;
        transition: all 0.3s ease;
        background: #f8f9fa;
    }
    .upload-area:hover {
        border-color: #007bff;
        background: #e9f7fe;
    }
    .upload-area.dragover {
        border-color: #007bff;
        background: #d1ecf1;
    }
    .csv-template {
        background: #f8f9fa;
        border-left: 4px solid #007bff;
    }
  </style>
</head>
<body class="bg-white">
<?php include __DIR__ . '/../includes/navbar.php'; ?>

<main class="rcp-main">
  <div class="container-fluid py-4">
    <div class="row mb-4">
      <div class="col-12">
        <h1 class="h3 mb-1 text-dark">User Management</h1>
        <p class="text-muted">Manage system users and permissions</p>
      </div>
    </div>

    <!-- Bulk Upload Section -->
    <div class="row mb-4">
      <div class="col-12">
        <div class="card border-0 shadow-sm bg-white">
          <div class="card-header bg-white border-bottom d-flex justify-content-between align-items-center">
            <h5 class="card-title mb-0 text-dark">
              <i class="fas fa-file-csv me-2"></i>Bulk User Upload
            </h5>
            <button class="btn btn-sm btn-outline-success" onclick="downloadTemplate()">
              <i class="fas fa-download me-1"></i>Download CSV Template
            </button>
          </div>
          <div class="card-body">
            <div class="row">
              <div class="col-md-6">
                <div class="upload-area" id="uploadArea">
                  <i class="fas fa-cloud-upload-alt fa-3x text-muted mb-3"></i>
                  <h5 class="text-dark">Drag & Drop CSV File</h5>
                  <p class="text-muted">or click to browse</p>
                  <input type="file" id="csvFile" accept=".csv" class="d-none">
                  <button type="button" class="btn btn-primary mt-2" onclick="document.getElementById('csvFile').click()">
                    <i class="fas fa-folder-open me-2"></i>Choose File
                  </button>
                  <div class="mt-3" id="fileInfo"></div>
                </div>
              </div>
              <div class="col-md-6">
                <div class="csv-template p-3">
                  <h6 class="text-dark"><i class="fas fa-info-circle me-2"></i>CSV Format Requirements</h6>
                  <small class="text-muted">
                    <ul class="mb-2">
                      <li>File must be in CSV format</li>
                      <li>Include header row with: full_name,email,username,password,role,status,department</li>
                      <li>Roles: user, admin, or superuser</li>
                      <li>Status: active or inactive</li>
                      <li>Department is optional</li>
                    </ul>
                  </small>
                  <button type="button" class="btn btn-sm btn-outline-info" onclick="previewTemplate()">
                    <i class="fas fa-eye me-1"></i>Preview Template
                  </button>
                </div>
                
                <div class="mt-3">
                  <button type="button" class="btn btn-success w-100" id="uploadBtn" onclick="uploadCSV()" disabled>
                    <i class="fas fa-upload me-2"></i>Upload Users
                  </button>
                  <div class="progress mt-2 d-none" id="uploadProgress">
                    <div class="progress-bar" role="progressbar" style="width: 0%"></div>
                  </div>
                </div>
              </div>
            </div>
            
            <div id="uploadResults" class="mt-3 d-none">
              <!-- Results will be displayed here -->
            </div>
          </div>
        </div>
      </div>
    </div>

    <!-- Add User Section -->
    <div class="row mb-4">
      <div class="col-12">
        <h4 class="mb-3 text-dark"><i class="fas fa-user-plus me-2"></i>Add New User</h4>
      </div>
    </div>
    
    <div class="row g-4">
      <!-- Add User Card -->
      <div class="col-lg-5">
        <div class="card border-0 shadow-sm bg-white">
          <div class="card-header bg-white border-bottom">
            <h5 class="card-title mb-0 text-dark">Create New User Account</h5>
          </div>
          <div class="card-body">
            <form id="userForm">
              <div class="mb-3">
                <label for="userFullName" class="form-label text-dark">Full Name *</label>
                <input type="text" class="form-control" id="userFullName" name="full_name" required>
              </div>
              
              <div class="mb-3">
                <label for="userEmail" class="form-label text-dark">Email Address *</label>
                <input type="email" class="form-control" id="userEmail" name="email" required>
              </div>
              
              <div class="mb-3">
                <label for="userUsername" class="form-label text-dark">Username *</label>
                <input type="text" class="form-control" id="userUsername" name="username" required>
              </div>
              
              <div class="mb-3">
                <label for="userPassword" class="form-label text-dark">Password *</label>
                <input type="password" class="form-control" id="userPassword" name="password" required>
              </div>
              
              <div class="row g-2">
                <div class="col-md-6">
                  <label for="userRole" class="form-label text-dark">Role *</label>
                  <select class="form-select" id="userRole" name="role" required>
                    <option value="user">User</option>
                    <option value="admin">Admin</option>
                    <option value="superuser">Superuser</option>
                  </select>
                </div>
                <div class="col-md-6">
                  <label for="userStatus" class="form-label text-dark">Status *</label>
                  <select class="form-select" id="userStatus" name="status" required>
                    <option value="active">Active</option>
                    <option value="inactive">Inactive</option>
                  </select>
                </div>
              </div>
              
              <div class="mb-3">
                <label for="userUnit" class="form-label text-dark">Department</label>
                <input type="text" class="form-control" id="userUnit" name="unit_name" placeholder="Enter department (optional)">
              </div>
              
              <button type="submit" class="btn btn-primary w-100">
                <i class="fas fa-user-plus me-2"></i>Create User
              </button>
            </form>
          </div>
        </div>
      </div>
      
      <!-- Users List Section -->
      <div class="col-lg-7">
        <div class="row mb-3">
          <div class="col-12">
            <h4 class="mb-3 text-dark"><i class="fas fa-users me-2"></i>System Users</h4>
          </div>
        </div>
        <div class="card border-0 shadow-sm bg-white">
          <div class="card-header bg-white border-bottom d-flex justify-content-between align-items-center">
            <h5 class="card-title mb-0 text-dark">User Accounts</h5>
            <button class="btn btn-sm btn-outline-primary" onclick="loadUsers()">
              <i class="fas fa-sync-alt me-1"></i>Refresh
            </button>
          </div>
          <div class="card-body">
            <div class="table-responsive">
              <table class="table table-hover">
                <thead class="table-light">
                  <tr>
                    <th class="text-dark">Name</th>
                    <th class="text-dark">Email</th>
                    <th class="text-dark">Role</th>
                    <th class="text-dark">Department</th>
                    <th class="text-dark">Status</th>
                    <th width="120" class="text-end text-dark">Actions</th>
                  </tr>
                </thead>
                <tbody id="usersTable">
                  <tr>
                    <td colspan="6" class="text-center py-4">
                      <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                      </div>
                    </td>
                  </tr>
                </tbody>
              </table>
            </div>
          </div>
        </div>
      </div>
    </div>
  </div>
</main>

<!-- Edit User Modal -->
<div class="modal fade" id="editUserModal" tabindex="-1">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header bg-white">
        <h5 class="modal-title text-dark">Edit User</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <form id="editUserForm">
        <div class="modal-body">
          <input type="hidden" id="editUserId" name="id">
          
          <div class="mb-3">
            <label for="editFullName" class="form-label text-dark">Full Name *</label>
            <input type="text" class="form-control" id="editFullName" name="full_name" required>
          </div>
          
          <div class="mb-3">
            <label for="editEmail" class="form-label text-dark">Email Address *</label>
            <input type="email" class="form-control" id="editEmail" name="email" required>
          </div>
          
          <div class="mb-3">
            <label for="editUsername" class="form-label text-dark">Username *</label>
            <input type="text" class="form-control" id="editUsername" name="username" required>
          </div>
          
          <div class="mb-3">
            <label for="editPassword" class="form-label text-dark">Password (leave blank to keep current)</label>
            <input type="password" class="form-control" id="editPassword" name="password" placeholder="Leave blank to keep current password">
          </div>
          
          <div class="row g-2">
            <div class="col-md-6">
              <label for="editRole" class="form-label text-dark">Role *</label>
              <select class="form-select" id="editRole" name="role" required>
                <option value="user">User</option>
                <option value="admin">Admin</option>
                <option value="superuser">Superuser</option>
              </select>
            </div>
            <div class="col-md-6">
              <label for="editStatus" class="form-label text-dark">Status *</label>
              <select class="form-select" id="editStatus" name="status" required>
                <option value="active">Active</option>
                <option value="inactive">Inactive</option>
              </select>
            </div>
          </div>
          
          <div class="mb-3">
            <label for="editUnit" class="form-label text-dark">Department</label>
            <input type="text" class="form-control" id="editUnit" name="unit_name" placeholder="Enter department (optional)">
          </div>
        </div>
        <div class="modal-footer bg-white">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
          <button type="submit" class="btn btn-primary">Save Changes</button>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteUserModal" tabindex="-1">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header bg-white">
        <h5 class="modal-title text-dark">Confirm Delete</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <p>Are you sure you want to delete user: <strong id="deleteUserName" class="text-dark">Unknown User</strong>?</p>
        <p class="text-danger">This action cannot be undone.</p>
      </div>
      <div class="modal-footer bg-white">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-danger" id="confirmDelete">Delete User</button>
      </div>
    </div>
  </div>
</div>

<!-- Template Preview Modal -->
<div class="modal fade" id="templateModal" tabindex="-1">
  <div class="modal-dialog modal-lg">
    <div class="modal-content">
      <div class="modal-header bg-white">
        <h5 class="modal-title text-dark">CSV Template Preview</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <div class="table-responsive">
          <table class="table table-bordered">
            <thead class="table-light">
              <tr>
                <th>full_name</th>
                <th>email</th>
                <th>username</th>
                <th>password</th>
                <th>role</th>
                <th>status</th>
                <th>department</th>
              </tr>
            </thead>
            <tbody>
              <tr>
                <td>John Doe</td>
                <td>john.doe@company.com</td>
                <td>johndoe</td>
                <td>TempPass123</td>
                <td>user</td>
                <td>active</td>
                <td>Sales</td>
              </tr>
              <tr>
                <td>Jane Smith</td>
                <td>jane.smith@company.com</td>
                <td>janesmith</td>
                <td>TempPass456</td>
                <td>admin</td>
                <td>active</td>
                <td>Marketing</td>
              </tr>
            </tbody>
          </table>
        </div>
        <div class="alert alert-info">
          <strong>Note:</strong> The first row must contain the header names exactly as shown above.
        </div>
      </div>
      <div class="modal-footer bg-white">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
        <button type="button" class="btn btn-success" onclick="downloadTemplate()">
          <i class="fas fa-download me-2"></i>Download Template
        </button>
      </div>
    </div>
  </div>
</div>

<script>
// CSV Bulk Upload Functionality
let selectedFile = null;

function setupCSVUpload() {
    const uploadArea = document.getElementById('uploadArea');
    const fileInput = document.getElementById('csvFile');
    
    // Drag and drop functionality
    uploadArea.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadArea.classList.add('dragover');
    });
    
    uploadArea.addEventListener('dragleave', () => {
        uploadArea.classList.remove('dragover');
    });
    
    uploadArea.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadArea.classList.remove('dragover');
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            handleFileSelect(files[0]);
        }
    });
    
    // File input change
    fileInput.addEventListener('change', (e) => {
        if (e.target.files.length > 0) {
            handleFileSelect(e.target.files[0]);
        }
    });
}

function handleFileSelect(file) {
    if (file.type !== 'text/csv' && !file.name.endsWith('.csv')) {
        alert('Please select a CSV file.');
        return;
    }
    
    selectedFile = file;
    document.getElementById('fileInfo').innerHTML = `
        <div class="alert alert-success">
            <i class="fas fa-check-circle me-2"></i>
            <strong>File selected:</strong> ${file.name}<br>
            <small>Size: ${(file.size / 1024).toFixed(2)} KB</small>
        </div>
    `;
    document.getElementById('uploadBtn').disabled = false;
}

function downloadTemplate() {
    const csvContent = "full_name,email,username,password,role,status,department\n" +
                      "John Doe,john.doe@company.com,johndoe,TempPass123,user,active,Sales\n" +
                      "Jane Smith,jane.smith@company.com,janesmith,TempPass456,admin,active,Marketing\n" +
                      "Bob Johnson,bob.johnson@company.com,bobjohnson,TempPass789,superuser,active,IT";
    
    const blob = new Blob([csvContent], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'user_upload_template.csv';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    window.URL.revokeObjectURL(url);
}

function previewTemplate() {
    const templateModal = new bootstrap.Modal(document.getElementById('templateModal'));
    templateModal.show();
}

async function uploadCSV() {
    if (!selectedFile) {
        alert('Please select a CSV file first.');
        return;
    }
    
    const uploadBtn = document.getElementById('uploadBtn');
    const progressBar = document.getElementById('uploadProgress');
    const progress = progressBar.querySelector('.progress-bar');
    const resultsDiv = document.getElementById('uploadResults');
    
    uploadBtn.disabled = true;
    progressBar.classList.remove('d-none');
    progress.style.width = '0%';
    
    const formData = new FormData();
    formData.append('csv_file', selectedFile);
    
    try {
        const response = await fetch('<?= WEB_BASE ?>api/users/bulk_upload.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        progress.style.width = '100%';
        
        if (result.success) {
            resultsDiv.innerHTML = `
                <div class="alert alert-success">
                    <h6><i class="fas fa-check-circle me-2"></i>Bulk Upload Completed</h6>
                    <p><strong>Success:</strong> ${result.data.success_count} users created</p>
                    ${result.data.errors.length > 0 ? `
                        <p><strong>Errors:</strong> ${result.data.errors.length} users failed</p>
                        <div class="mt-2">
                            <button class="btn btn-sm btn-outline-danger" type="button" 
                                    onclick="showUploadErrors(${JSON.stringify(result.data.errors).replace(/"/g, '&quot;')})">
                                View Errors
                            </button>
                        </div>
                    ` : ''}
                </div>
            `;
            resultsDiv.classList.remove('d-none');
            
            // Reload users list
            loadUsers();
            
        } else {
            resultsDiv.innerHTML = `
                <div class="alert alert-danger">
                    <h6><i class="fas fa-exclamation-triangle me-2"></i>Upload Failed</h6>
                    <p>${result.message}</p>
                </div>
            `;
            resultsDiv.classList.remove('d-none');
        }
        
    } catch (error) {
        resultsDiv.innerHTML = `
            <div class="alert alert-danger">
                <h6><i class="fas fa-exclamation-triangle me-2"></i>Upload Error</h6>
                <p>${error.message}</p>
            </div>
        `;
        resultsDiv.classList.remove('d-none');
    } finally {
        uploadBtn.disabled = false;
        setTimeout(() => {
            progressBar.classList.add('d-none');
        }, 2000);
    }
}

function showUploadErrors(errors) {
    let errorHtml = '<ul>';
    errors.forEach(error => {
        errorHtml += `<li><strong>Row ${error.row}:</strong> ${error.message}</li>`;
    });
    errorHtml += '</ul>';
    
    alert('Upload Errors:\n\n' + errors.map(e => `Row ${e.row}: ${e.message}`).join('\n'));
}

// Real API integration
async function loadUsers() {
    const tbody = document.getElementById('usersTable');
    tbody.innerHTML = `
        <tr>
            <td colspan="6" class="text-center py-4">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
            </td>
        </tr>
    `;
    
    try {
        const response = await fetch('<?= WEB_BASE ?>api/users/read.php');
        const result = await response.json();
        
        if (result.success && result.data.length > 0) {
            let html = '';
            result.data.forEach(user => {
                const statusBadge = user.status === 'active' ? 
                    '<span class="badge bg-success">Active</span>' : 
                    '<span class="badge bg-secondary">Inactive</span>';
                
                const roleBadge = user.role === 'superuser' ? 
                    '<span class="badge bg-warning text-dark">Superuser</span>' : 
                    user.role === 'admin' ? 
                    '<span class="badge bg-info">Admin</span>' : 
                    '<span class="badge bg-primary">User</span>';
                
                html += `
                <tr>
                    <td class="text-dark">${escapeHtml(user.full_name)}</td>
                    <td class="text-dark">${escapeHtml(user.email)}</td>
                    <td class="text-dark">${roleBadge}</td>
                    <td class="text-dark">${escapeHtml(user.department || 'N/A')}</td>
                    <td class="text-dark">${statusBadge}</td>
                    <td class="text-end">
                        <button class="btn btn-sm btn-outline-primary me-1" onclick="editUser(${user.id})" title="Edit User">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-danger" onclick="showDeleteModal(${user.id}, '${escapeHtml(user.full_name)}')" title="Delete User">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>
                </tr>`;
            });
            tbody.innerHTML = html;
        } else {
            tbody.innerHTML = `
                <tr>
                    <td colspan="6" class="text-center py-4 text-muted">
                        <i class="fas fa-users me-2"></i>
                        No users found. Create your first user above.
                    </td>
                </tr>`;
        }
    } catch (error) {
        tbody.innerHTML = `
            <tr>
                <td colspan="6" class="text-center py-4 text-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    Error loading users: ${error.message}
                </td>
            </tr>`;
    }
}

function escapeHtml(text) {
    if (text === null || text === undefined) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

let currentEditUserId = null;
let currentDeleteUserId = null;

// Edit User Functionality
async function editUser(userId) {
    try {
        const response = await fetch('<?= WEB_BASE ?>api/users/read.php');
        const result = await response.json();
        
        if (result.success) {
            const user = result.data.find(u => u.id == userId);
            if (user) {
                // Populate the edit form
                document.getElementById('editUserId').value = user.id;
                document.getElementById('editFullName').value = user.full_name;
                document.getElementById('editEmail').value = user.email;
                document.getElementById('editUsername').value = user.username;
                document.getElementById('editRole').value = user.role;
                document.getElementById('editStatus').value = user.status;
                document.getElementById('editUnit').value = user.department || '';
                document.getElementById('editPassword').value = '';
                
                currentEditUserId = userId;
                
                // Show the modal
                const editModal = new bootstrap.Modal(document.getElementById('editUserModal'));
                editModal.show();
            }
        }
    } catch (error) {
        alert('Error loading user data: ' + error.message);
    }
}

// Delete User Functionality
function showDeleteModal(userId, userName) {
    currentDeleteUserId = userId;
    document.getElementById('deleteUserName').textContent = userName;
    
    const deleteModal = new bootstrap.Modal(document.getElementById('deleteUserModal'));
    deleteModal.show();
}

async function deleteUser(userId) {
    try {
        const response = await fetch('<?= WEB_BASE ?>api/users/delete.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ id: userId })
        });
        
        const result = await response.json();
        
        if (result.success) {
            // Close the modal
            const deleteModal = bootstrap.Modal.getInstance(document.getElementById('deleteUserModal'));
            deleteModal.hide();
            
            // Show success message
            alert('User deleted successfully!');
            
            // Reload the user list
            loadUsers();
        } else {
            alert('Error deleting user: ' + result.message);
        }
    } catch (error) {
        alert('Error deleting user: ' + error.message);
    }
}

// Form submission handlers
document.addEventListener('DOMContentLoaded', function() {
    setupCSVUpload();
    loadUsers();
    
    // Create User Form
    document.getElementById('userForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const data = Object.fromEntries(formData);
        
        try {
            const response = await fetch('<?= WEB_BASE ?>api/users/create.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(data)
            });
            
            const result = await response.json();
            
            if (result.success) {
                alert('User created successfully!');
                this.reset();
                loadUsers();
            } else {
                alert('Error: ' + result.message);
            }
        } catch (error) {
            alert('Error creating user: ' + error.message);
        }
    });
    
    // Edit User Form
    document.getElementById('editUserForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const data = Object.fromEntries(formData);
        
        // Remove password field if empty
        if (!data.password) {
            delete data.password;
        }
        
        try {
            const response = await fetch('<?= WEB_BASE ?>api/users/update.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(data)
            });
            
            const result = await response.json();
            
            if (result.success) {
                // Close the modal
                const editModal = bootstrap.Modal.getInstance(document.getElementById('editUserModal'));
                editModal.hide();
                
                alert('User updated successfully!');
                loadUsers();
            } else {
                alert('Error: ' + result.message);
            }
        } catch (error) {
            alert('Error updating user: ' + error.message);
        }
    });
    
    // Delete Confirmation
    document.getElementById('confirmDelete').addEventListener('click', function() {
        if (currentDeleteUserId) {
            deleteUser(currentDeleteUserId);
        }
    });
});
</script>

<?php include __DIR__ . '/../includes/footer.php'; ?>
</body>
</html>